%% Monte-Carlo simulation of migration-assimilation dynamic
% with indefinite end---continue until none has incentive to move.
% Record the final aggregate state variables (no record of dynamic process)
% Keep each agent's initial preference & action, 
% while changing the parameter

tic;
clear;
close all

%%%%%%%%%%%%%%%%
%% parameters
%%%%%%%%%%%%%%%%
%Choose which parameter to randomize
para_rand='vx'

%benchmark values
T=1000 %the _maximum_ period of each simulation
TT=200 %total number of running simulations

l=0.05 %speed of preference evolution
q=0.1 %probability to obtain an opportunity to switch the community
% r=0

U0=0 % Constant exogeneous utility on living in city
uPop=0 % uPop*#[agents in the community]=pop-dependent exogenous utility in each community

M1=999 % no. of the type 1 agents
M2=999 % no. of the type 2 agents
H1=667 % no. of the type 1 agents in the hometown
H2=667 % no. of the type 2 agents in the hometown

LPrf1=-1 % lower bound of type 1 agents' preference
UPrf1=0 % upper bound of type 1 agents' preference
LPrf2=0 % lower bound of type 2 agents' preference
UPrf2=1 % upper bound of type 2 agents' preference
Type1Name='type -'
Type2Name='type +'

switch para_rand
    case 'lambda'
        paramtlabel='\lambda'; paramtexp='Speed of preference evolution';
    case 'mu'
        paramtlabel='\mu'; paramtexp='Frequency of community switching opportunity';
%     case 'r'
%         paramtlabel='r'; paramtexp='Probability of exogenous relocation';
    case 'v0'
        paramtlabel='v^0'; paramtexp='Constant exogeneous utility on living in city';
    case 'vx'
        paramtlabel='v_x'; paramtexp='Rate of proportional increase in utility with #agents in each community';
    case 'HtoM'
        paramtlabel='HtoM'; paramtexp='proportion of hometown residents among each type';
    case 'H2toM2'
        paramtlabel='H2toM2'; paramtexp=['proportion of hometown residents among ' Type2Name];
end

%%%%%%%
%% Setting up the files
%%%%%%%
% File location and names
starttime=datestr(now,'yymmddHHMM'); %Use the starting time for file names
ResultFolder=sprintf('./SimResults/MC%s_%s',starttime,para_rand) %The folder to save the results
mkdir(ResultFolder)
ResultXLS=[ResultFolder '/' sprintf('MC%s_result.xlsx',starttime)]; %The file location of the Excel spread sheet
DiaryTXT=[ResultFolder '/' sprintf('MC%s_diary.txt',starttime)]; %The file location of the diary text file

%Start recording the log on the diary file
diary(DiaryTXT)
disp('Monte-Carlo simulation of migration-assimilation dynamic with indefinite ending period')
disp('Ver. 1608')
disp(['Randomized parameter: ' paramtlabel ' ,' paramtexp])

%%%%%%%%%%%%%%%%%%%
%% Individual agents' initial preference matrix
%%%%%%%%%%%%%%%%%%%

Prf0=[(UPrf1-LPrf1)*rand(M1,1)+LPrf1*ones(M1,1);(UPrf2-LPrf2)*rand(M2,1)+LPrf2*ones(M2,1)]; 
	% The initial preference matrix: size (M1+M2)*1, i.e., a column vector
	% The first M1 cells are the type-1 agents' preferences, and the rest M2 cells are the type-2s'.
	% Throughout the simulation, each column of a (M1+M2) row vector corresponds to each agent.
	% Here a type-i agent's preference are drawn from the uniform distribution over [Li,Ui].

Type=[1*ones(M1,1);2*ones(M2,1)];
    % The type (belonging population group) of each agent

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Run the simulation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Prepare output variables
paramt=zeros(TT,1);EndPd=ones(TT,1);
Nm=zeros(TT,8);AvePrf0=zeros(TT,8);AvePrfT=zeros(TT,8);

for m=1:TT %start the loop: one loop m corresponds to one simulation of T-period dynamic

	%% Set the initial values
    
    %%------------
    % Generate parameters for Monte Carlo
    %-----------
    switch para_rand
        case 'lambda'
            l=rand; paramt(m,1)=l; %speed of preference assimilation. "\lambda" in the paper. To randomize, put "rand"
        case 'mu'
            q=rand; paramt(m,1)=q; %revision rate of community
        case 'r'
            r=q*rand; paramt(m,1)=r; %probability of exogenous relocation
        case 'v0'
            U0=-0.1+0.2*rand; paramt(m,1)=U0;
        case 'vx'
            uPop=-0.5+rand; paramt(m,1)=uPop;
        case 'HtoM'
            H=round(rand*[M1 M2]);H1=H(1);H2=H(2); paramt(m,1)=H1/M1;
        case 'H2toM2'
            H2=round((0.1+0.8*rand)*M2); paramt(m,1)=H2/M2;
    end

    Com0=[1*ones(H1,1);0*ones(M1-H1+M2-H2,1);2*ones(H2,1)]; %Notice that there is no randomization.
    % A2 is the agents' initial community matrix (1: hometown for type 1, 0:city, 2: hometown for 2)
    % Among M1 type-1 agents, the first H1 agents are initially located at their hometown and the rest are at city.
    % Among M2 type-1 agents, the last H2 agents are initially located at their hometown and the rest are at city.    

	%G1H; %Number of type 1 in the hometown at the end
	%G1C; %Number of type 1 in the city at the end
	%G2H; %Number of type 2 in the hometown at the end
	%G2C; %Number of type 2 in the city at the end

    Prf_t=Prf0;Com_t=Com0;OptCom_t=Com0;
    AvePrfC_t=[mean(Prf_t(Com_t==0));mean(Prf_t(Com_t==1));mean(Prf_t(Com_t==2))]; %Calculate the average preference in each community

	% Run the dynamic
	for t=2:T+1 % each i corresponds to one period
		
        %%----------------------------
        % Agents switch communities
        %----------------------------
        % Threshold on preference for the choice between hometown and city
        U0_t=U0+uPop*(sum(Com_t==0)/(M1+M2));U1_t=uPop*(sum(Com_t==1)/(M1+M2));U2_t=uPop*(sum(Com_t==2)/(M1+M2));
        if AvePrfC_t(1)==AvePrfC_t(2)
            if U0_t>U1_t
                th1_t=LPrf1;
            elseif U0_t<U1_t
                th1_t=UPrf1;
            end
        else
            th1_t=0.5*(AvePrfC_t(1)+AvePrfC_t(2)+(U0_t-U1_t)/(AvePrfC_t(2)-AvePrfC_t(1)));% threshold 1 between being better in town 1 and being better in city
        end
        if AvePrfC_t(1)==AvePrfC_t(3)
            if U0_t>U2_t
                th2_t=UPrf2;
            elseif U0_t<U2_t
                th2_t=LPrf2;
            end
        else
            th2_t=0.5*(AvePrfC_t(1)+AvePrfC_t(3)+(U0_t-U2_t)/(AvePrfC_t(3)-AvePrfC_t(1)));% threshold 2 between being better in town 2 and being better in city
        end
     
        %Optimal community for each agent
        OptCom_t(1:M1)=0*(Prf_t(1:M1)>th1_t)+1*(Prf_t(1:M1)<th1_t)+(Prf_t(1:M1)==th1_t).*Com_t(1:M1);
        OptCom_t(M1+1:M1+M2)=0*(Prf_t(M1+1:M1+M2)<th2_t)+2*(Prf_t(M1+1:M1+M2)>th2_t)+(Prf_t(M1+1:M1+M2)==th2_t).*Com_t(M1+1:M1+M2);
        
        %End the simulation if everyone has no incentive to move
        if Com_t==OptCom_t; 
            Prf_t=AvePrfC_t(Com_t+ones(M1+M2,1)); % Make the preference to the limit under no switch in communities
            break
        end
        
        % Voluntary switches
		Rev_t=binornd(1,q,M1+M2,1); %community switching vector. 
			%If Rev(k)=1, agent k can switch the community.
			%q is the probability to get such an opportunity.
        Com_t=Rev_t.*OptCom_t+(ones(M1+M2,1)-Rev_t).*Com_t;
%         % Exogenous relocation
%         Reloc_t=rand(M1+M2,1); %Forced relocation vector
%             %If Reloc_t(k)<0.5*r, agent k is relocated to city.
%             %If 0.5*r<=Reloc_t(k)<r, agent k is relocated to its hometown.
%             %Otherwise, agent k stays at the new community.
%         Com_t(1:M1)=0*(Reloc_t(1:M1)<0.5*r)+1*(Reloc_t(1:M1)>=0.5*r & Reloc_t(1:M1)<r)+(Reloc_t(1:M1)>=r).*Com_t(1:M1);
%         Com_t(M1+1:M1+M2)=0*(Reloc_t(M1+1:M1+M2)<0.5*r)+2*(Reloc_t(M1+1:M1+M2)>=0.5*r & Reloc_t(M1+1:M1+M2)<r)+(Reloc_t(M1+1:M1+M2)>=r).*Com_t(M1+1:M1+M2);
        
        %%-------------------- 
        % Preference evolution
        %---------------------
        AvePrfC_t=[mean(Prf_t(Com_t==0));mean(Prf_t(Com_t==1));mean(Prf_t(Com_t==2))]; %Calculate the average preference in each community
        Prf_t=l*AvePrfC_t(Com_t+ones(M1+M2,1))+(1-l)*Prf_t; %Preference evolution
    end %Ends the dynamic
        %Maybe it is better to check whether no agents have incentive to switch community further.
    %% Record the state variables in the final period
    Com_T=Com_t; Prf_T=Prf_t; clear Com_t OptCom_t Prf_t AvePrfC_t Rev_t th1_t th2_t; 
    EndPd(m,1)=t;
    %Categorize agents
    I111=find(Type==1 & Com0==1 & Com_T==1); %Type-1, Born in town 1, End in town 1;
    I110=find(Type==1 & Com0==1 & Com_T==0); %Type-1, Born in town 1, End in city;
    I101=find(Type==1 & Com0==0 & Com_T==1); %Type-1, Born in city, End in town 1;
    I100=find(Type==1 & Com0==0 & Com_T==0); %Type-1, Born in city, End in city;
    I200=find(Type==2 & Com0==0 & Com_T==0); %Type-2, Born in city, End in city;
    I202=find(Type==2 & Com0==0 & Com_T==2); %Type-2, Born in city, End in town 2;
    I220=find(Type==2 & Com0==2 & Com_T==0); %Type-2, Born in town 2, End in city;
    I222=find(Type==2 & Com0==2 & Com_T==2); %Type-2, Born in town 2, End in town 2;
    
    %Aggregate
    I11x=[I111;I110]; %Type-1, Born in town 1
    I10x=[I101;I100]; %Type-1, Born in city
    I1x1=[I101;I111]; %Type-1, End in town 1
    I1x0=[I100;I110]; %Type-1, End in city
    I22x=[I222;I220]; %Type-2, Born in town 2
    I20x=[I202;I200]; %Type-2, Born in city
    I2x2=[I202;I222]; %Type-2, End in town 2
    I2x0=[I200;I220]; %Type-2, End in city
    
    %Count #agents in each category
    Nm(m,:)=[length(I111),length(I110),length(I101),length(I100),...%
        length(I200),length(I202),length(I220),length(I222)];
    NmAgg(m,:)=[length(I11x),length(I10x),length(I1x1),length(I1x0),...%
        length(I22x),length(I20x),length(I2x2),length(I2x0)];
    
    %Average preference at the beginning
    AvePrf0(m,:)=[mean(Prf0(I111)),mean(Prf0(I110)),mean(Prf0(I101)),mean(Prf0(I100)),...%
        mean(Prf0(I200)),mean(Prf0(I202)),mean(Prf0(I220)),mean(Prf0(I222))];
    AvePrf0Agg(m,:)=[mean(Prf0(I11x)),mean(Prf0(I10x)),mean(Prf0(I1x1)),mean(Prf0(I1x0)),...%
        mean(Prf0(I22x)),mean(Prf0(I20x)),mean(Prf0(I2x2)),mean(Prf0(I2x0)),mean(Prf0(Type==1)),mean(Prf0(Type==2))];
    
    %Average preference at the end
    AvePrfT(m,:)=[mean(Prf_T(I111)),mean(Prf_T(I110)),mean(Prf_T(I101)),mean(Prf_T(I100)),...%
        mean(Prf_T(I200)),mean(Prf_T(I202)),mean(Prf_T(I220)),mean(Prf_T(I222))];
    AvePrfTAgg(m,:)=[mean(Prf_T(I11x)),mean(Prf_T(I10x)),mean(Prf_T(I1x1)),mean(Prf_T(I1x0)),...%
        mean(Prf_T(I22x)),mean(Prf_T(I20x)),mean(Prf_T(I2x2)),mean(Prf_T(I2x0)),mean(Prf_T(Type==1)),mean(Prf_T(Type==2))];
    
       %MATLAB recommends to define the entire matrix and keep its size
        % and the memory allocation unchanged.
	%Result columns follow the sequence as: (H1-H1 means: born at hometown 1 and
	% evetually at hometown 1, C means city.)
	%Random value of lamda #Average Preference at H1 #AP at city #AP at H2 #H1-H1 #H1-C #C-H1 #C-C #C-H2 #H2-C #H2-H2 
    APDiff=AvePrfT-AvePrf0;APDiffSoc=sum(APDiff.*Nm/(M1+M2),2);APDiff1=sum(APDiff(:,1:4).*Nm(:,1:4)/M1,2);APDiff2=sum(APDiff(:,5:8).*Nm(:,5:8)/M2,2);
    DinD=APDiff-[APDiff1*ones(1,4) APDiff2*ones(1,4)];

end
diary off

figure; plot(paramt,EndPd,'bx')
set(gca, 'FontName','Ariel','FontSize',18);  
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('Ending period of simulation','FontName','Ariel','FontSize',24);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_EndPd',starttime)],'Period for all agents to reach zero incentive to switch')

figure; plot(paramt,NmAgg(:,1)/(M1+M2),'rx',paramt,NmAgg(:,2)/(M1+M2),'ro',paramt,NmAgg(:,3)/(M1+M2),'bx',paramt,NmAgg(:,4)/(M1+M2),'bo')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Born in Town','Born in City','End in Town','End in City'); 
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('Population share','FontName','Ariel','FontSize',24);ylim([0 M1/(M1+M2)]);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_ShareAgg1',starttime)],['Population share: ' Type1Name])

figure; plot(paramt,NmAgg(:,5)/(M1+M2),'rx',paramt,NmAgg(:,6)/(M1+M2),'ro',paramt,NmAgg(:,7)/(M1+M2),'bx',paramt,NmAgg(:,8)/(M1+M2),'bo')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Born in Town','Born in City','End in Town','End in City');
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('Population share','FontName','Ariel','FontSize',24);ylim([0 M2/(M1+M2)]);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_ShareAgg2',starttime)],['Population share: ' Type2Name])

figure; plot(paramt,AvePrf0Agg(:,1),'rx',paramt,AvePrf0Agg(:,2),'ro',paramt,AvePrf0Agg(:,9),'r.',...
    paramt,AvePrfTAgg(:,3),'bx',paramt,AvePrfTAgg(:,4),'bo',paramt,AvePrfTAgg(:,9),'b.')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Town, initial','City, initial','All, initial','Town, end','City, end','All, end');
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('preference','FontName','Ariel','FontSize',24);
   ylimvec=ylim;ylim([min([ylimvec(1) LPrf1]) max([ylimvec(2) UPrf1])]); ylimvec=ylim;set(gca,'YTick',unique([ylimvec(1) LPrf1 (LPrf1+UPrf1)/2 UPrf1 ylimvec(2) 0]));clear ylimvec;box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_AvePrfAgg1',starttime)],['average preference: ' Type1Name])

figure; plot(paramt,AvePrf0Agg(:,5),'rx',paramt,AvePrf0Agg(:,6),'ro',paramt,AvePrf0Agg(:,10),'r.',...
    paramt,AvePrfTAgg(:,7),'bx',paramt,AvePrfTAgg(:,8),'bo',paramt,AvePrfTAgg(:,10),'b.')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Town, initial','City, initial','All, initial','Town, end','City, end','All, end');
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('preference','FontName','Ariel','FontSize',24);
   ylimvec=ylim;ylim([min([ylimvec(1) LPrf2]) max([ylimvec(2) UPrf2])]); ylimvec=ylim;set(gca,'YTick',unique([ylimvec(1) LPrf2 (LPrf2+UPrf2)/2 UPrf2 ylimvec(2) 0]));clear ylimvec;box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_AvePrfAgg2',starttime)],['average preference: ' Type2Name])

figure; plot(paramt,Nm(:,1)/(M1+M2),'rx',paramt,Nm(:,2)/(M1+M2),'ro',paramt,Nm(:,3)/(M1+M2),'bo',paramt,Nm(:,4)/(M1+M2),'bx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Town->Town','Town->City','City->Town','City->City'); 
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('Population share','FontName','Ariel','FontSize',24);ylim([0 M1/(M1+M2)]);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_Share1',starttime)],['Population share at the last period: ' Type1Name])

figure; plot(paramt,Nm(:,5)/(M1+M2),'bx',paramt,Nm(:,6)/(M1+M2),'bo',paramt,Nm(:,7)/(M1+M2),'ro',paramt,Nm(:,8)/(M1+M2),'rx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('City->City','City->Town','Town->City','Town->Town');
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('Population share','FontName','Ariel','FontSize',24);ylim([0 M2/(M1+M2)]);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_Share2',starttime)],['Population share at the last period: ' Type2Name])

figure; plot(paramt,AvePrf0(:,1),'rx',paramt,AvePrf0(:,2),'ro',paramt,AvePrf0(:,3),'bo',paramt,AvePrf0(:,4),'bx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Town->Town','Town->City','City->Town','City->City');
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('preference','FontName','Ariel','FontSize',24);
   ylimvec=ylim;ylim([min([ylimvec(1) LPrf1]) max([ylimvec(2) UPrf1])]); ylimvec=ylim;set(gca,'YTick',unique([ylimvec(1) LPrf1 (LPrf1+UPrf1)/2 UPrf1 ylimvec(2) 0]));clear ylimvec;box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_AvePrf01',starttime)],['Average preference at the first period: ' Type1Name])

figure; plot(paramt,AvePrf0(:,5),'bx',paramt,AvePrf0(:,6),'bo',paramt,AvePrf0(:,7),'ro',paramt,AvePrf0(:,8),'rx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('City->City','City->Town','Town->City','Town->Town');
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('preference','FontName','Ariel','FontSize',24);
   ylimvec=ylim;ylim([min([ylimvec(1) LPrf2]) max([ylimvec(2) UPrf2])]); ylimvec=ylim;set(gca,'YTick',unique([ylimvec(1) LPrf2 (LPrf2+UPrf2)/2 UPrf2 ylimvec(2) 0]));clear ylimvec;box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_AvePrf02',starttime)],['Average preference at the first period: ' Type2Name])

figure; plot(paramt,AvePrfT(:,1),'rx',paramt,AvePrfT(:,2),'ro',paramt,AvePrfT(:,3),'bo',paramt,AvePrfT(:,4),'bx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Town->Town','Town->City','City->Town','City->City'); xlabel([paramtlabel ': ' paramtexp]);
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('preference','FontName','Ariel','FontSize',24);
   ylimvec=ylim;ylim([min([ylimvec(1) LPrf1]) max([ylimvec(2) UPrf1])]); ylimvec=ylim;set(gca,'YTick',unique([ylimvec(1) LPrf1 (LPrf1+UPrf1)/2 UPrf1 ylimvec(2) 0]));clear ylimvec;box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_AvePrfT1',starttime)],['Average preference in the last period: ' Type1Name])

figure; plot(paramt,AvePrfT(:,5),'bx',paramt,AvePrfT(:,6),'bo',paramt,AvePrfT(:,7),'ro',paramt,AvePrfT(:,8),'rx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('City->City','City->Town','Town->City','Town->Town');
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('preference','FontName','Ariel','FontSize',24);
   ylimvec=ylim;ylim([min([ylimvec(1) LPrf2]) max([ylimvec(2) UPrf2])]); ylimvec=ylim;set(gca,'YTick',unique([ylimvec(1) LPrf2 (LPrf2+UPrf2)/2 UPrf2 ylimvec(2) 0]));clear ylimvec;box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_AvePrfT2',starttime)],['Average preference in the last period: ' Type2Name])

figure; plot(paramt,APDiff(:,1),'rx',paramt,APDiff(:,2),'ro',paramt,APDiff(:,3),'bo',paramt,APDiff(:,4),'bx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Town->Town','Town->City','City->Town','City->City'); 
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('preference change','FontName','Ariel','FontSize',24);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_ApDiff1',starttime)],['Change in average preference: ' Type1Name])

figure; plot(paramt,APDiff(:,5),'bx',paramt,APDiff(:,6),'bo',paramt,APDiff(:,7),'ro',paramt,APDiff(:,8),'rx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('City->City','City->Town','Town->City','Town->Town'); 
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('preference change','FontName','Ariel','FontSize',24);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_ApDiff2',starttime)],['Change in average preference: ' Type2Name])

figure; plot(paramt,DinD(:,1),'rx',paramt,DinD(:,2),'ro',paramt,DinD(:,3),'bo',paramt,DinD(:,4),'bx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('Town->Town','Town->City','City->Town','City->City'); 
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('difference in preference change','FontName','Ariel','FontSize',24);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_DinD1',starttime)],['Difference in preference compared to average in the type: ' Type1Name])

figure; plot(paramt,DinD(:,5),'bx',paramt,DinD(:,6),'bo',paramt,DinD(:,7),'ro',paramt,DinD(:,8),'rx')
set(gca, 'FontName','Ariel','FontSize',18);  
legend('City->City','City->Town','Town->City','Town->Town'); 
xlabel([paramtlabel ': ' paramtexp],'FontName','Ariel','FontSize',20);
ylabel('difference in preference change','FontName','Ariel','FontSize',24);box on
FigJpegPrint([ResultFolder '/' sprintf('MC%s_DinD2',starttime)],['Difference in preference compared to average in the type: ' Type2Name])

xlswrite(ResultXLS,{paramtlabel, 'EndPd'},'EndPd');
xlswrite(ResultXLS,[paramt,EndPd],'EndPd','A2');
xlswrite(ResultXLS,{paramtlabel, 'Nm_1Tx','Nm_1Cx','Nm_1xT','Nm_1xC','Nm_2Tx','Nm_2Cx','Nm_2xT','Nm_2xC'},'NumbersAgg');
xlswrite(ResultXLS,[paramt,Nm],'NumbersAgg','A2');
xlswrite(ResultXLS,{paramtlabel, 'Nm_1TT','Nm_1TC','Nm_1CT','Nm_1CC','Nm_2CC','Nm_2CT','Nm_2TC','Nm_2TT'},'NumbersBreak');
xlswrite(ResultXLS,[paramt,Nm],'NumbersBreak','A2');
xlswrite(ResultXLS,{paramtlabel, 'AP0_1Tx','AP0_1Cx','AP0_1xT','AP0_1xC','AP0_2Tx','AP0_2Cx','AP0_2xT','AP0_2xC','AP0_1','AP0_2'...
    'APT_1Tx','APT_1Cx','APT_1xT','APT_1xC','APT_2Tx','APT_2Cx','APT_2xT','APT_2xC','APT_1','APT_2'},'AvePrefAgg');
xlswrite(ResultXLS,[paramt,AvePrf0Agg,AvePrfTAgg],'AvePrefAgg','A2');
xlswrite(ResultXLS,{paramtlabel, 'AP0_1TT','AP0_1TC','AP0_1CT','AP0_1CC','AP0_2CC','AP0_2CT','AP0_2TC','AP0_2TT',...
    'APT_1TT','APT_1TC','APT_1CT','APT_1CC','APT_2CC','APT_2CT','APT_2TC','APT_2TT'},'AvePrefBreak');
xlswrite(ResultXLS,[paramt,AvePrf0,AvePrfT],'AvePrefBreak','A2');
xlswrite(ResultXLS,{paramtlabel, 'APDiff_1TT','APDiff_1TC','APDiff_1CT','APDiff_1CC','APDiff_2CC','APDiff_2CT','APDiff_2TC','APDiff_2TT','APDiff_All','APDiff_1','APDiff_2'},'AvePrefChange');
xlswrite(ResultXLS,[paramt,APDiff,APDiffSoc,APDiff1,APDiff2],'AvePrefChange','A2');
xlswrite(ResultXLS,{paramtlabel, 'DinD_1TT','DinD_1TC','DinD_1CT','DinD_1CC','DinD_2CC','DinD_2CT','DinD_2TC','DinD_2TT'},'DinD');
xlswrite(ResultXLS,[paramt,DinD],'DinD','A2');

diary(DiaryTXT)
elapsedTime=toc;
disp(sprintf('Finished in %5.2f seconds.',elapsedTime))
diary off

%% History of revisions
% 1st version 2014/2/8 by Futao
% 2014/02/11 Dai added comments.
% 2014/02/16 Ver.3 Dai
%    use matrix operation in best response,
%    add headers to Excel file, plot the transitions
% 2014/05/24 Ver.5. Dai
%   add exogeneous relocation
% 2016/08/04 Ver.1608 Dai
%   Change the date format from mmddHHMM to yymmddHHMM
%   Add constant and lienar term to utility
% 2016/08/10 Ver.IndefEnd1608 Dai
%   Change the ending period to be indefinite (with T reinterpreted as the upper bound)
