%% Single-shot simulation of migration-assimilation dynamic
% Record the transition of individual preference & community


tic;
clear;
cd 
close all
starttime=datestr(now,'yymmddHHMM');
ResultFolder=sprintf('./SimResults/Single%s',starttime)
mkdir(ResultFolder)
ResultXLS=[ResultFolder '/' sprintf('Single%s_result.xlsx',starttime)];
DiaryTXT=[ResultFolder '/' sprintf('Single%s_diary.txt',starttime)];

diary(DiaryTXT)

%%%%%%%%%%%%%%%%
%% parameters
%%%%%%%%%%%%%%%%
%For simulation
T=100 %the period of each simulation
M1=1000 % no. of the type 1 agents
M2=1000 % no. of the type 2 agents
H1=500 % no. of the type 1 agents in the hometown
H2=500 % no. of the type 2 agents in the hometown

LPrf1=-1 % lower bound of type 1 agents' preference
UPrf1=0 % upper bound of type 1 agents' preference
LPrf2=0 % lower bound of type 2 agents' preference
UPrf2=1 % upper bound of type 2 agents' preference

U0=0 % Constant exogeneous utility on living in city
uPop=0 % uPop*#[agents in the community]=pop-dependent exogenous utility in each community

l=0.05 %speed of preference assimilation. "\lambda" in the paper.
q=0.1 %probability to obtain an opportunity to switch the community
r=0 %probability of exgenous relocation

%For plots
DetailedPlot=1 % Set 1 to plot individual preference evolution by community at EACH period
%p1=0; p2=0; p3=0

%%%%%%%%%%%%%%%%%%%
%% Individual agents' initial preference/action matrix
%%%%%%%%%%%%%%%%%%%

Prf0=[(UPrf1-LPrf1)*rand(M1,1)+LPrf1*ones(M1,1);(UPrf2-LPrf2)*rand(M2,1)+LPrf2*ones(M2,1)]; 
	% The initial preference matrix: size (M1+M2)*1, i.e., a column vector
	% The first M1 cells are the type-1 agents' preferences, and the rest M2 cells are the type-2s'.
	% Throughout the simulation, each column of a (M1+M2) row vector corresponds to each agent.
	% Here a type-i agent's preference are drawn from the uniform distribution over [Li,Ui].

Type=[1*ones(M1,1);2*ones(M2,1)];
    % The type (belonging population group) of each agent
    
Com0=[1*ones(H1,1);0*ones(M1-H1+M2-H2,1);2*ones(H2,1)];
    % A2 is the agents' initial community matrix (1: hometown for type 1, 0:city, 2: hometown for 2)
    % Among M1 type-1 agents, the first H1 agents are initially located at their hometown and the rest are at city.
    % Among M2 type-1 agents, the last H2 agents are initially located at their hometown and the rest are at city.    

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Run the simulation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Prepare recording variables to keep the history of each variable
PrfH=zeros(T+1,M1+M2);ComH=zeros(T+1,M1+M2);thH=zeros(T,2);

%% Set the initial values
Prf_t=Prf0; Com_t=Com0;
PrfH(1,:)=Prf_t';ComH(1,:)=Com_t';
AvePrfC_t=[mean(Prf_t(Com_t==0));mean(Prf_t(Com_t==1));mean(Prf_t(Com_t==2))]; %Calculate the average preference in each community

%G1H; %Number of type 1 in the hometown at the end
%G1C; %Number of type 1 in the city at the end
%G2H; %Number of type 2 in the hometown at the end
%G2C; %Number of type 2 in the city at the end

%% Run the dynamic
for t=2:T+1 % each i corresponds to one period

    %%----------------------------
    % Agents switch communities
    %----------------------------
    U0_t=U0+uPop*(sum(Com_t==0)/(M1+M2));U1_t=uPop*(sum(Com_t==1)/(M1+M2));U2_t=uPop*(sum(Com_t==2)/(M1+M2));
    
    if AvePrfC_t(1)==AvePrfC_t(2)
        if U0_t>U1_t
            th1_t=LPrf1;
        elseif U0_t<U1_t
            th1_t=UPrf1;
        end
    else
        th1_t=0.5*(AvePrfC_t(1)+AvePrfC_t(2)+(U0_t-U1_t)/(AvePrfC_t(2)-AvePrfC_t(1)));% threshold 1 between being better in town 1 and being better in city
    end
    if AvePrfC_t(1)==AvePrfC_t(3)
        if U0_t>U2_t
            th2_t=UPrf2;
        elseif U0_t<U2_t
            th2_t=LPrf2;
        end
    else
        th2_t=0.5*(AvePrfC_t(1)+AvePrfC_t(3)+(U0_t-U2_t)/(AvePrfC_t(3)-AvePrfC_t(1)));% threshold 2 between being better in town 2 and being better in city
    end

    % Voluntary switches
    Rev_t=binornd(1,q,M1+M2,1); %community switching vector. 
        %If Rev(k)=1, agent k can switch the community.
        %q is the probability to get such an opportunity.
    Com_t(1:M1)=Rev_t(1:M1).*(0*(Prf_t(1:M1)>th1_t)+1*(Prf_t(1:M1)<th1_t))+(ones(M1,1)-Rev_t(1:M1)+Rev_t(1:M1).*(Prf_t(1:M1)==th1_t)).*Com_t(1:M1);
    Com_t(M1+1:M1+M2)=Rev_t(M1+1:M1+M2).*(0*(Prf_t(M1+1:M1+M2)<th2_t)+2*(Prf_t(M1+1:M1+M2)>th2_t))+(ones(M2,1)-Rev_t(M1+1:M1+M2)+Rev_t(M1+1:M1+M2).*(Prf_t(M1+1:M1+M2)==th2_t)).*Com_t(M1+1:M1+M2);
    % Exogenous relocation
    Reloc_t=rand(M1+M2,1); %Forced relocation vector
        %If Reloc_t(k)<0.5*r, agent k is relocated to city.
        %If 0.5*r<=Reloc_t(k)<r, agent k is relocated to its hometown.
        %Otherwise, agent k stays at the new community.
    Com_t(1:M1)=0*(Reloc_t(1:M1)<0.5*r)+1*(Reloc_t(1:M1)>=0.5*r & Reloc_t(1:M1)<r)+(Reloc_t(1:M1)>=r).*Com_t(1:M1);
    Com_t(M1+1:M1+M2)=0*(Reloc_t(M1+1:M1+M2)<0.5*r)+2*(Reloc_t(M1+1:M1+M2)>=0.5*r & Reloc_t(M1+1:M1+M2)<r)+(Reloc_t(M1+1:M1+M2)>=r).*Com_t(M1+1:M1+M2);

    %%-------------------- 
    % Preference evolution
    %---------------------
    AvePrfC_t=[mean(Prf_t(Com_t==0));mean(Prf_t(Com_t==1));mean(Prf_t(Com_t==2))]; %Calculate the average preference in each community
    Prf_t=l*AvePrfC_t(Com_t+ones(M1+M2,1))+(1-l)*Prf_t; %Preference evolution
   
    %%-------------------- 
    % keep the record
    %---------------------
    thH(t-1,:)=[th1_t,th2_t]; %thresholds should be compared to the preference in the end of the last pd.
    PrfH(t,:)=Prf_t';ComH(t,:)=Com_t';
end %Ends the dynamic
    %Maybe it is better to check whether no agents have incentive to
    % switch community further.

%% Record the state variables in the final period
Com_T=Com_t; Prf_T=Prf_t; clear Com_t Prf_t AvePrfC_t Rev_t th1_t th2_t; 

figure; hold on
plot([1:T+1],PrfH(:,Type==1),'r.',[1:T+1],PrfH(:,Type==2),'b.');
%legend(p1,'Type 1','Type 2');
plot(thH,'k','Linewidth',1.5);box on
xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
ylabel('preference','FontName','Ariel','FontSize',12);ylim([-1 1]);
FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Prf_All',starttime)],'Individual preference evolution')

figure; hold on
plot([1:T+1],PrfH(:,Type==1 & Com0==1 & Com_T==1),'r.',[1:T+1],PrfH(:,Type==2 & Com0==2 &  Com_T==2),'b.');
%legend(p2,'Type 1 and end in Town 1','Type 2 and end in Town 2');
plot(thH,'k','Linewidth',1.5);box on
xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
ylabel('preference','FontName','Ariel','FontSize',12);ylim([-1 1]);
FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Prf_TT',starttime)],'Individual preference evolution: hometowns to hometowns')

figure; hold on
plot([1:T+1],PrfH(:,Type==1 & Com0==0 & Com_T==1),'r.',[1:T+1],PrfH(:,Type==2 & Com0==0 &  Com_T==2),'b.');
%legend(p2,'Type 1 and end in Town 1','Type 2 and end in Town 2');
plot(thH,'k','Linewidth',1.5);box on
xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
ylabel('preference','FontName','Ariel','FontSize',12);ylim([-1 1]);
FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Prf_CT',starttime)],'Individual preference evolution: city to hometowns');

figure; hold on
plot([1:T+1],PrfH(:,Type==1 & Com0==0 & Com_T==0),'r.',[1:T+1],PrfH(:,Type==2 & Com0==0 &  Com_T==0),'b.');
plot(thH,'k','Linewidth',1.5);box on
xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
ylabel('preference','FontName','Ariel','FontSize',12);ylim([-1 1]);
FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Prf_CC',starttime)],'Individual preference evolution: city to city');

figure; hold on
plot([1:T+1],PrfH(:,Type==1 & Com0==1 & Com_T==0),'r.',[1:T+1],PrfH(:,Type==2 & Com0==2 &  Com_T==0),'b.');
plot(thH,'k','Linewidth',1.5);box on
xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
ylabel('preference','FontName','Ariel','FontSize',12);ylim([-1 1]);
FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Prf_TC',starttime)],'Individual preference evolution: hometown to city');

figure;hold on
%plot(sum(ComH==1,2)/(M1+M2),'r-','Linewidth',1.5);plot(sum(ComH==2,2)/(M1+M2),'b-','Linewidth',1.5);plot(sum(ComH==0,2)/(M1+M2),'g-','Linewidth',1.5);
%legend('Town 1','City','Town 2');
%b=bar([sum(ComH==1,2)/(M1+M2),sum(ComH==0,2)/(M1+M2),sum(ComH==2,2)/(M1+M2)],1,'stacked','LineStyle','none')
%b(1).FaceColor='r';b(2).FaceColor='g';b(3).FaceColor='b';
plot(sum(ComH==1,2)/(M1+M2),'r-','Linewidth',1.5);plot(1-sum(ComH==2,2)/(M1+M2),'b-','Linewidth',1.5);box on
xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
ylabel('Population share','FontName','Ariel','FontSize',12);ylim([0 1]);
FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Share',starttime)],'Population share of each community');

figure; hold on %This takes much time due to the iteration.
for i=1:T+1
    plot(i,mean(PrfH(i,ComH(i,:)==1)),'r*');
    plot(i,mean(PrfH(i,ComH(i,:)==2)),'b*');
    plot(i,mean(PrfH(i,ComH(i,:)==0)),'g*');
end
plot(T+1,mean(PrfH(T+1,ComH(T+1,:)==1)),'r*',T+1,mean(PrfH(T+1,ComH(T+1,:)==2)),'b*',...
    T+1,mean(PrfH(T+1,ComH(T+1,:)==0)),'g*');
%legend(p5,'Town 1','Town 2','City');
plot(thH,'k','Linewidth',1.5);box on
xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
ylabel('preference','FontName','Ariel','FontSize',12);ylim([-1 1]);
FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Prf_Ave',starttime)],'Evolution of average preference in each community');

if DetailedPlot==1 %Draw time-taking graphs
    % figure; hold on %This takes much time due to the iteration.
    % for i=1:T
    %     plot(i,PrfH(i,ComH(i,:)==1),'r.');
    %     plot(i,PrfH(i,ComH(i,:)==2),'b.');
    %     plot(i,PrfH(i,ComH(i,:)==0),'g.');
    % end
    % plot(T+1,PrfH(T+1,ComH(T+1,:)==1),'r.',T+1,PrfH(T+1,ComH(T+1,:)==2),'b.',...
    %     T+1,PrfH(T+1,ComH(T+1,:)==0),'g.');
    % %legend(p4,'Town 1','Town 2','City');
    % plot(thH,'k','Linewidth',1.5);
    % title('Individual preference evolution: color by community in each pd');xlabel('period');
    % saveas(gcf,sprintf('Single%s_Prf_ByCom.fig',starttime));

    figure; hold on %This takes much time due to the iteration.
    for i=1:T
        plot(i,PrfH(i,ComH(i,:)==1),'r.');
        plot(i,PrfH(i,ComH(i,:)==2),'b.');
    end
    plot(T+1,PrfH(T+1,ComH(T+1,:)==1),'r.',T+1,PrfH(T+1,ComH(T+1,:)==2),'b.');
    %legend(p4,'Town 1','Town 2','City');
    plot(thH,'k','Linewidth',1.5);box on
    xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
    ylabel('preference','FontName','Ariel','FontSize',12);ylim([-1 1]);
    FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Prf_ByCom_Towns',starttime)],'Individual preference evolution: color by community in each pd');

    figure; hold on %This takes much time due to the iteration.
    for i=1:T
        plot(i,PrfH(i,ComH(i,:)==0),'g.');
    end
    plot(T+1,PrfH(T+1,ComH(T+1,:)==0),'g.');
    %legend(p4,'Town 1','Town 2','City');
    plot(thH,'k','Linewidth',1.5);box on
    xlabel('period','FontName','Ariel','FontSize',12);xlim([1 T+1]);
    ylabel('preference','FontName','Ariel','FontSize',12);ylim([-1 1]);
    FigJpegPrintNoLeg([ResultFolder '/' sprintf('Single%s_Prf_ByCom_City',starttime)],'Individual preference evolution: color by community in each pd');
end



% 
% %Categorize agents
% I111=find(Type==1 & Com0==1 & Com_T==1); %Type-1, Born in town 1, End in town 1;
% I110=find(Type==1 & Com0==1 & Com_T==0); %Type-1, Born in town 1, End in city;
% I101=find(Type==1 & Com0==0 & Com_T==1); %Type-1, Born in city, End in town 1;
% I100=find(Type==1 & Com0==0 & Com_T==0); %Type-1, Born in city, End in city;
% I200=find(Type==2 & Com0==0 & Com_T==0); %Type-2, Born in city, End in city;
% I202=find(Type==2 & Com0==0 & Com_T==2); %Type-2, Born in city, End in town 2;
% I220=find(Type==2 & Com0==2 & Com_T==0); %Type-2, Born in town 2, End in city;
% I222=find(Type==2 & Com0==2 & Com_T==2); %Type-2, Born in town 2, End in town 2;
% 
% %Count #agents in each category
% Nm(m,:)=[length(I111),length(I110),length(I101),length(I100),...%
%     length(I200),length(I202),length(I220),length(I222)];
% 
% %Average preference at the beginning
% AvePrf0(m,:)=[mean(Prf0(I111)),mean(Prf0(I110)),mean(Prf0(I101)),mean(Prf0(I100)),...%
%     mean(Prf0(I200)),mean(Prf0(I202)),mean(Prf0(I220)),mean(Prf0(I222))];
% 
% %Average preference at the end
% AvePrfT(m,:)=[mean(Prf_T(I111)),mean(Prf_T(I110)),mean(Prf_T(I101)),mean(Prf_T(I100)),...%
%     mean(Prf_T(I200)),mean(Prf_T(I202)),mean(Prf_T(I220)),mean(Prf_T(I222))];
% 
%    %MATLAB recommends to define the entire matrix and keep its size
%     % and the memory allocation unchanged.
% %Result columns follow the sequence as: (H1-H1 means: born at hometown 1 and
% % evetually at hometown 1, C means city.)
% %Random value of lamda #Average Preference at H1 #AP at city #AP at H2 #H1-H1 #H1-C #C-H1 #C-C #C-H2 #H2-C #H2-H2 
% 
% diary off
% xlswrite(ResultXLS,{'lambda', 'Nm_1TT','Nm_1TC','Nm_1CT','Nm_1CC','Nm_2CC','Nm_2CT','Nm_2TC','Nm_2TT'},'Numbers');
% xlswrite(ResultXLS,[lambda,Nm],'Numbers','A2');
% xlswrite(ResultXLS,{'lambda', 'AP0_1TT','AP0_1TC','AP0_1CT','AP0_1CC','AP0_2CC','AP0_2CT','AP0_2TC','AP0_2TT',...
%     'APT_1TT','APT_1TC','APT_1CT','APT_1CC','APT_2CC','APT_2CT','APT_2TC','APT_2TT'},'AvePref');
% xlswrite(ResultXLS,[lambda,AvePrf0,AvePrfT],'AvePref','A2');
% APDiff=AvePrfT-AvePrf0;APDiffSoc=sum(APDiff.*Nm/(M1+M2),2);APDiff1=sum(APDiff(:,1:4).*Nm(:,1:4)/M1,2);APDiff2=sum(APDiff(:,5:8).*Nm(:,5:8)/M2,2);
% xlswrite(ResultXLS,{'lambda', 'APDiff_1TT','APDiff_1TC','APDiff_1CT','APDiff_1CC','APDiff_2CC','APDiff_2CT','APDiff_2TC','APDiff_2TT','APDiff_All','APDiff_1','APDiff_2'},'AvePrefChange');
% xlswrite(ResultXLS,[lambda,APDiff,APDiffSoc,APDiff1,APDiff2],'AvePrefChange','A2');
% DinD=APDiff-[APDiff1*ones(1,4) APDiff2*ones(1,4)];
% xlswrite(ResultXLS,{'lambda', 'DinD_1TT','DinD_1TC','DinD_1CT','DinD_1CC','DinD_2CC','DinD_2CT','DinD_2TC','DinD_2TT'},'DinD');
% xlswrite(ResultXLS,[lambda,DinD],'DinD','A2');
% 
% figure; plot(lambda,Nm(:,1),'rx',lambda,Nm(:,2),'r.',lambda,Nm(:,3),'b.',lambda,Nm(:,4),'bx')
% title('numbers of agents in the last period: type 1');legend('Town->Town','Town->City','City->Town','City->City'); xlabel('\lambda: speed of preference evolution');
% saveas(gcf,sprintf('Single%s_Nm1.fig',starttime));
% figure; plot(lambda,Nm(:,5),'bx',lambda,Nm(:,6),'b.',lambda,Nm(:,7),'r.',lambda,Nm(:,8),'rx')
% title('numbers of agents in the last period: type 2');legend('City->City','City->Town','Town->City','Town->Town'); xlabel('\lambda: speed of preference evolution');
% saveas(gcf,sprintf('Single%s_Nm2.fig',starttime));


diary(DiaryTXT)
elapsedTime=toc;
disp(sprintf('Finished in %5.2f seconds.',elapsedTime))
diary off

%% History
% 1st version 2014/02/18 by Dai
% based on MC_APfixed
% Ver. 1608 2016/08/05 by Dai
% based on MonteCarlo1608